## SEIR simulation plots

# Plots a cmparison of susceptible and infected populations 
# for multiple scenarios

# Uses seir_sim (below)

# User specifies a list of values for one of the key parameters:
#   R0 is the basic reproductive number
#   TL is the mean incubation time
#   TI is the mean infectious duration
#   I0 is the initial infectious fraction
#   V is the initial immune fraction

# The program is designed so that only a few lines need to be modified to
# make a new experiment (see '###' comments)
#     line ###1 defines the list of independent variable values
#     line ###2 links the independent variable name and values

# output figure:
#   left panel: susceptible fraction
#   right panel: infected fraction

# The file can be modified to experiment with other parameters.

# by Glenn Ledder
# written 2020/10/15
# revised 2020/10/29

# direct comments to gledder@unl.edu

### SEIR_simplot appears below!

#########################################

seir_sim <- function(beta,eta,gamma,E0,I0,V,target)
	{

	## DATA

	maxdays = 1000

	## FUNCTION FOR THE DIFFERENTIAL EQUATION

    	yprime <- function(yy)
		{
      	S = yy[1]
      	E = yy[2]
		I = yy[3]
      	Sp = -beta*S*I
      	Ep = -Sp-eta*E
		Ip = eta*E-gamma*I
      	yp = c(Sp,Ep,Ip)
		return(yp)
		}

	## INITIALIZATION

	# set up results data structure with Y=[S,E,I] and R

	results = matrix(0,nrow=maxdays+1,ncol=4)	
	S0 = 1-E0-I0-V
	Y = c(S0,E0,I0)
	R = V
	results[1,] = c(Y,R)

	oldI = I0

	## FUNCTION FOR rk4

      # y0 is a column vector of initial conditions at t
      # y is a column vector of values at t+dt

	rk4 <- function(dt,y0)
		{
		k1 = yprime(y0)
        	k2 = yprime(y0+0.5*dt*k1);
        	k3 = yprime(y0+0.5*dt*k2);
        	k4 = yprime(y0+dt*k3);
        	yy = y0+dt*(k1+2*k2+2*k3+k4)/6;
		return(yy)
		}

	## COMPUTATION

	for (t in 1:maxdays)
		{
		Y = rk4(1,Y)
		R = 1-sum(Y)
    		results[t+1,] = c(Y,R)
   		if (Y[2]+Y[3]>min(target,oldI))
			{
        		oldI = Y[2]+Y[3]
			} else {
	      	results = results[1:(t+1),]
        		break
			}
		}

	## OUTPUT

	return(results)

	## END

	}

#########################################

### SEIR_simplot starts here! ###

## DEFAULT SCENARIO DATA

R0 = 2.5
TL = 5
TI = 10
I0 = 0.001
V = 0

## INDEPENDENT VARIABLE DATA

### This section needs to be modified for each experiment.

###1 xvals is the set of independent variable values
xvals = c(5,4,3,2)

## COMMON DATA

target = 0.001
colors = c('#1f77b4','#d62728','#ff7f0e','#9467bd','#2ca02c','#17becf','#8c564b','#7f7f7f','#bcbd22','#e377c2')

## INITIALIZATION

par(mfrow=c(1,2))    # set the plotting area into a 1x2 array

N = length(xvals)

## COMPUTATION

for (n in 1:N)
	{

    	###2 The left side of this statement needs to be the independent
    	###2 variable for the experiment.
    	R0 = xvals[n]

    	eta = 1/TL;
    	gamma = 1/TI;
    	beta = gamma*R0;

	b = eta+gamma
	c = eta*(gamma-beta*(1-V))
    	lambda = 0.5*(sqrt(b^2-4*c)-b)
    	rho = (lambda+gamma)/eta
    	E0 = rho*I0

	results = seir_sim(beta,eta,gamma,E0,I0,V,target)
    	days = length(results[,1])-1

	if (n==1)
		{
		allresults = results
		rows = length(allresults[,1])
		allrows = rows
		alldays = days
		} else {
		allresults = rbind(allresults,results)
		rows = length(allresults[,1])
		allrows = c(allrows,rows)
		alldays = c(alldays,days)
		}
	}

## PLOTS

ylabel = c("susceptible","infectious")

tmax = max(alldays)
Imax = max(allresults[,3])
ymax = c(1,1)

for (j in 1:2)
	{
	n = 1
	column = 2*j-1
	x = 0:alldays[n]
	first = 1
	last = allrows[n]
	y = allresults[first:last,column]
	plot(x, y, type="l", col=colors[n], xlim=c(0,tmax), ylim=c(0,ymax[j]), xlab="days", ylab=ylabel[j], lwd=2, xaxs="i", yaxs="i")
	for (n in 2:N)
		{
		x = 0:alldays[n]
		first = allrows[n-1]+1
		last = allrows[n]
		y = allresults[first:last,column]
		lines(x, y, col=colors[n], lwd=2)
		}
	}

legend(150, 0.9, legend=c("R0=5","R0=4","R0=3","R0=2"),col=colors[1:4], lty=1, lwd=2)

